<?php
session_start();

$jsonFile = __DIR__ . "/data.json";

// domain default
$defaultDomain = $_SERVER['HTTP_HOST'] ?? "contoh.com";

// daftar domain (bisa ditambah manual di array)
$domains = [
    $defaultDomain,
    "contoh.com"
];

// load database JSON
if (!file_exists($jsonFile)) {
    file_put_contents($jsonFile, json_encode([]));
}
$data = json_decode(file_get_contents($jsonFile), true);

// pesan hasil proses
$message = [];

$isAjax = !empty($_SERVER['HTTP_X_REQUESTED_WITH']) && strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) === 'xmlhttprequest';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $urls = trim($_POST['urls'] ?? "");
    $slug = trim($_POST['slug'] ?? "");
    $domain = trim($_POST['domain'] ?? $defaultDomain);

    if ($urls !== "") {
        $urlArray = array_map('trim', explode("\n", $urls));
        foreach ($urlArray as $url) {
            if (!filter_var($url, FILTER_VALIDATE_URL)) {
                $message[] = ["error" => "URL tidak valid: $url"];
                continue;
            }
            // generate slug unik
            $id = $slug !== "" ? $slug : substr(md5(uniqid(mt_rand(), true)), 0, 8);
            if (isset($data[$id])) {
                $message[] = ["error" => "Slug sudah dipakai: $id"];
                continue;
            }
            // simpan ke database json
            $data[$id] = [
                "original" => $url,
                "domain"   => $domain,
                "created"  => date("Y-m-d H:i:s")
            ];
            $shortUrl = "https://$domain/e/$id";
            $message[] = ["success" => "Shortlink berhasil: <a href='$shortUrl' target='_blank'>$shortUrl</a>"];
        }
        file_put_contents($jsonFile, json_encode($data, JSON_PRETTY_PRINT));
    }

    if ($isAjax) {
        header('Content-Type: application/json');
        echo json_encode(['messages' => $message]);
        exit;
    }
}
?>
<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <meta name="description" content="Pemendek URL cepat, gratis, dan aman dengan opsi slug kustom dan domain.">
    <title>Pemendek URL - Pendekkan Tautan Anda</title>
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/tailwindcss@2.2.19/dist/tailwind.min.css">
    <link rel="stylesheet" href="https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600;700&display=swap">
    <link rel="icon" type="image/svg+xml" href="data:image/svg+xml,<svg xmlns='http://www.w3.org/2000/svg' viewBox='0 0 20 20' fill='%23007BFF'><path d='M10 2a8 8 0 100 16 8 8 0 000-16zm0 14a6 6 0 110-12 6 6 0 010 12zm-2-6a1 1 0 112 0v3a1 1 0 11-2 0v-3zm2-4a1 1 0 100-2 1 1 0 000 2z'/></svg>">
    <style>
        body {
            font-family: 'Inter', sans-serif;
        }
        .animate-slide-up {
            animation: slideUp 0.5s ease-out;
        }
        @keyframes slideUp {
            from { transform: translateY(20px); opacity: 0; }
            to { transform: translateY(0); opacity: 1; }
        }
        .hover-scale {
            transition: transform 0.2s ease;
        }
        .hover-scale:hover {
            transform: scale(1.02);
        }
    </style>
</head>
<body class="bg-gray-100 min-h-screen flex items-center justify-center">
    <div class="container max-w-2xl mx-auto p-6 bg-white rounded-2xl shadow-lg animate-slide-up">
        <h1 class="text-3xl font-bold text-center text-gray-800 mb-6">Pemendek URL</h1>
        <form id="shortenerForm" class="space-y-6">
            <div>
                <label class="block text-sm font-medium text-gray-700 mb-2">Pendekkan Massal (Maks 10 URL, satu URL per baris):</label>
                <textarea name="urls" rows="4" placeholder="https://contoh1.com&#10;https://contoh2.com" class="w-full p-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition"></textarea>
            </div>
            <div>
                <label class="block text-sm font-medium text-gray-700 mb-2">Slug Kustom Opsional (min 8 karakter):</label>
                <input type="text" name="slug" placeholder="contohslug123" class="w-full p-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition">
            </div>
            <div>
                <label class="block text-sm font-medium text-gray-700 mb-2">Pilih Domain:</label>
                <select name="domain" class="w-full p-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition">
                    <?php foreach ($domains as $d): ?>
                        <option value="<?= htmlspecialchars($d) ?>"><?= htmlspecialchars($d) ?></option>
                    <?php endforeach; ?>
                </select>
            </div>
            <button type="submit" class="w-full py-3 px-4 bg-blue-600 text-white font-semibold rounded-lg hover:bg-blue-700 transition hover-scale">Buat Shortlink(s)</button>
        </form>

        <div id="messageContainer" class="mt-6 space-y-4"></div>

        <div class="mt-8 p-6 bg-blue-50 rounded-lg hover-scale transition">
            <h3 class="text-lg font-semibold text-blue-700 mb-3">💡 Mengapa Menggunakan Pemendek URL Kami?</h3>
            <ul class="space-y-2 text-gray-600">
                <li class="flex items-start">
                    <svg class="w-5 h-5 text-blue-500 mr-2 mt-1" fill="currentColor" viewBox="0 0 20 20"><path fill-rule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zm3.707-9.293a1 1 0 00-1.414-1.414L9 10.586 7.707 9.293a1 1 0 00-1.414 1.414l2 2a1 1 0 001.414 0l4-4z" clip-rule="evenodd"/></svg>
                    <span><strong>Pendekkan URL Instan</strong> — Tempel URL target & klik tombol "Buat Shortlink" untuk memendekkannya seketika. Tidak ada penundaan atau timer.</span>
                </li>
                <li class="flex items-start">
                    <svg class="w-5 h-5 text-blue-500 mr-2 mt-1" fill="currentColor" viewBox="0 0 20 20"><path fill-rule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zm3.707-9.293a1 1 0 00-1.414-1.414L9 10.586 7.707 9.293a1 1 0 00-1.414 1.414l2 2a1 1 0 001.414 0l4-4z" clip-rule="evenodd"/></svg>
                    <span><strong>Gratis, Cepat & Aman</strong> — Tidak ada registrasi, pendaftaran, login, akun, rencana, kartu, pembayaran, periode uji coba.</span>
                </li>
                <li class="flex items-start">
                    <svg class="w-5 h-5 text-blue-500 mr-2 mt-1" fill="currentColor" viewBox="0 0 20 20"><path fill-rule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zm3.707-9.293a1 1 0 00-1.414-1.414L9 10.586 7.707 9.293a1 1 0 00-1.414 1.414l2 2a1 1 0 001.414 0l4-4z" clip-rule="evenodd"/></svg>
                    <span><strong>Opsi Domain Berganda (Segera Hadir)</strong> — Pilih dari berbagai domain untuk membuat shortlink bermerek yang sesuai dengan kebutuhan Anda.</span>
                </li>
            </ul>
        </div>
    </div>
    <script>
        document.getElementById('shortenerForm').addEventListener('submit', function(e) {
            e.preventDefault();
            const formData = new FormData(this);
            const messageContainer = document.getElementById('messageContainer');
            messageContainer.innerHTML = '';

            fetch(window.location.href, {
                method: 'POST',
                body: formData,
                headers: {
                    'X-Requested-With': 'XMLHttpRequest'
                }
            })
            .then(response => response.json())
            .then(data => {
                const messages = data.messages || [];
                let allLinks = [];
                messages.forEach(m => {
                    const div = document.createElement('div');
                    if (m.success) {
                        div.className = 'p-4 bg-green-100 text-green-700 rounded-lg flex items-center justify-between space-x-2';
                        const successText = document.createElement('span');
                        successText.innerHTML = m.success;
                        div.appendChild(successText);

                        const link = successText.querySelector('a').href;
                        allLinks.push(link);

                        const copyBtn = document.createElement('button');
                        copyBtn.textContent = 'Salin';
                        copyBtn.className = 'ml-2 py-1 px-2 bg-blue-500 text-white rounded hover:bg-blue-600 transition';
                        copyBtn.onclick = function() {
                            navigator.clipboard.writeText(link).then(() => {
                                copyBtn.textContent = 'Disalin!';
                                setTimeout(() => copyBtn.textContent = 'Salin', 2000);
                            });
                        };
                        div.appendChild(copyBtn);
                    } else if (m.error) {
                        div.className = 'p-4 bg-red-100 text-red-700 rounded-lg flex items-center space-x-2';
                        const errorText = document.createElement('span');
                        errorText.textContent = m.error;
                        div.appendChild(errorText);
                    }
                    messageContainer.appendChild(div);
                });

                if (allLinks.length > 1) {
                    const copyAllBtn = document.createElement('button');
                    copyAllBtn.textContent = 'Salin Semua';
                    copyAllBtn.className = 'mt-4 w-full py-2 px-4 bg-green-600 text-white font-semibold rounded-lg hover:bg-green-700 transition';
                    copyAllBtn.onclick = function() {
                        navigator.clipboard.writeText(allLinks.join('\n')).then(() => {
                            copyAllBtn.textContent = 'Semua Disalin!';
                            setTimeout(() => copyAllBtn.textContent = 'Salin Semua', 2000);
                        });
                    };
                    messageContainer.appendChild(copyAllBtn);
                }
            })
            .catch(error => {
                console.error('Error:', error);
                const errorDiv = document.createElement('div');
                errorDiv.className = 'p-4 bg-red-100 text-red-700 rounded-lg';
                errorDiv.textContent = 'Terjadi kesalahan saat memproses permintaan.';
                messageContainer.appendChild(errorDiv);
            });
        });
    </script>
</body>
</html>